/////////////////////////////////////////////////////////////
// CINEMA 4D SDK                                           //
/////////////////////////////////////////////////////////////
// (c) 1989-2006 MAXON Computer GmbH, all rights reserved  //
/////////////////////////////////////////////////////////////

#ifndef _GEDYNAMICARRAY_H_
#define _GEDYNAMICARRAY_H_

#include "c4d_file.h"
#include "ge_math.h"
#include "c4d_memory.h"
#include "c4d_baselist.h"
#include "c4d_general.h"
#include "ge_lmatrix.h"
#include "ge_lvector.h"
#ifdef __API_INTERN__
	#include "hyperfile.h"
#endif

#if defined __LINUX
#include <new>
#elif defined __MAC
#include <new>
#else
#include <new.h>
#endif

#include <string.h>

#define DATYPE_CHAR			-1
#define DATYPE_UCHAR		-2
#define DATYPE_WORD			-3
#define DATYPE_UWORD		-4
#define DATYPE_LONG			-5
#define DATYPE_ULONG		-6
#define DATYPE_REAL			-7
#define DATYPE_LREAL		-8
#define DATYPE_BOOL			-9
#define DATYPE_TIME			-10
#define DATYPE_VECTOR		-11
#define DATYPE_LVECTOR	-12
#define DATYPE_MATRIX		-13
#define DATYPE_LMATRIX	-14
#define DATYPE_STRING		-15
#define DATYPE_FILENAME -16
#define DATYPE_LLONG		-17

//dynamic array template
template<class TYPE> class GeDynamicArray
{
protected:
	TYPE*									ptr;
	TYPE*									l_ptr;

	VLONG									l_cnt;
	VLONG									totcnt;
	VLONG									blocksize;
public:
	GeDynamicArray();
	GeDynamicArray(VLONG size);
	GeDynamicArray(VLONG size, VLONG block);
	GeDynamicArray(const GeDynamicArray<TYPE> &g);
	~GeDynamicArray(void);

	typedef void	GeDynamicArrayCallback(VLONG index, TYPE *value, void *data);
	VLONG	count;

	TYPE*					Release(void);
	void					Free(void);
	GeDynamicArray<TYPE> &operator =(const GeDynamicArray<TYPE> &t);
	GeDynamicArray<TYPE> &operator =(const TYPE *t);
	friend	Bool operator !(const GeDynamicArray<TYPE> &t) { return t.count == 0; }
	operator TYPE * ()											{	return l_ptr;	}
	TYPE & operator[]	( LONG i ) const			{	return l_ptr[i];	}
#if defined __LINUX && !defined __C4D_64BIT
	TYPE & operator[] ( VLONG i ) const   { return l_ptr[i]; }
#endif
	TYPE & operator[]	( LLONG i ) const			{	return l_ptr[i];	}
#if !defined __LINUX && !(defined __MAC && __LP64__)
	TYPE & operator[] ( int i )	const				{	return operator[]((LONG) i);	}
#endif
	TYPE & operator[] ( unsigned i ) const	{	return operator[]((LONG) i);	}
	TYPE * const *operator &()							{	return &l_ptr;	}
	VLONG					GetCount(void) const			{ return count; }
	void					SetBlockSize(VLONG size);
	VLONG					GetBlockSize(void) const	{ return blocksize; }
	Bool					Read(HyperFile *hf, GeDynamicArray<LONG> &typelist);
	Bool					Write(HyperFile *hf, GeDynamicArray<LONG> &typelist);
	void					DoToAll(GeDynamicArrayCallback *callback, void *data);
	void					DoToSome(GeDynamicArrayCallback *callback, void *data, VLONG do_start, VLONG do_end);
	void					DoToSearch(GeDynamicArrayCallback *callback, void *data, TYPE p);
	//Sets the size without actually resizing the data (be careful)
	void					SetDummySize(VLONG size)		{ count = totcnt = size; }
	Bool					ReScopeNC(VLONG cnt, Bool make_new = TRUE);
	Bool					ReScope(VLONG cnt);
	Bool					ReSizeNC(VLONG cnt, Bool make_new = TRUE);
	Bool					ReSize(VLONG cnt);
	Bool					ReSize(VLONG c_from, VLONG c_to);
	inline void		Fill(VLONG c_from, VLONG c_to, TYPE t);
	inline void		Fill(TYPE t);
	Bool					Insert(TYPE p, VLONG index);
	Bool					Insert(const TYPE *p, VLONG index, VLONG cnt);
	Bool					MergeIn(GeDynamicArray<TYPE>&p,VLONG index);
	Bool					InsertSpace(VLONG index, VLONG amount, Bool clear = TRUE, Bool makenew = TRUE);
	inline Bool		Push(TYPE p);
	Bool					Remove(VLONG index, VLONG amount = 1);
	inline void		RemoveAll(TYPE p);
	void					Pop(void);
	inline VLONG	Find(TYPE p);
	inline VLONG	Find(TYPE p, VLONG index);
	void					Shift(VLONG index, VLONG cnt);
	void					ShiftBlock(VLONG b_from, VLONG b_to, VLONG amount);
	void					Grab(TYPE *p, VLONG amount);
};

template<class TYPE> GeDynamicArray<TYPE>::GeDynamicArray(void)
{
	ptr = NULL;
	blocksize = 4;
	Free();
}

template<class TYPE> GeDynamicArray<TYPE>::GeDynamicArray(VLONG size) 
{
	ptr = NULL;
	blocksize = 4;
	Free();
	ReScope(size); 
}

template<class TYPE> GeDynamicArray<TYPE>::GeDynamicArray(VLONG size, VLONG block) 
{ 
	ptr = NULL;
	blocksize = block; 
	Free();
	ReScope(size); 
}

template<class TYPE> GeDynamicArray<TYPE>::GeDynamicArray(const GeDynamicArray<TYPE> &g) 
{ 
	ptr = NULL;
	blocksize = g.blocksize; 
	Free();
	*this = g;
}


template<class TYPE> GeDynamicArray<TYPE>::~GeDynamicArray(void)
{
	Free();
}

template<class TYPE> TYPE* GeDynamicArray<TYPE>::Release(void)
{ 
	memmove(ptr,l_ptr,sizeof(TYPE)*count);
	blocksize = 4;
	VLONG cnt = count;
	count += l_cnt;
	l_ptr = ptr;
	l_cnt = 0;
	ReSizeNC(cnt);
	TYPE	*tmp = ptr;
	ptr = NULL; 
	l_ptr = NULL;
	totcnt = 0; 
	count = 0; 
	l_cnt = 0;
	return tmp;
}

template<class TYPE> void GeDynamicArray<TYPE>::Free(void)
{ 
	if (ptr) 
	{
		VLONG i=0;
		TYPE *buf=l_ptr;

		for (i=0;i<count;i++,buf++) buf->~TYPE();

		GeFree(ptr);
	}
	ptr = NULL;
	l_ptr = NULL;
	count = 0;
	totcnt = 0;
	l_cnt = 0;
}

template<class TYPE> GeDynamicArray<TYPE>& GeDynamicArray<TYPE>::operator =(const GeDynamicArray<TYPE> &t)
{
	if (t.blocksize != blocksize) 
	{
		Free();
		SetBlockSize(t.blocksize);
	}
	if (!ReScopeNC(t.count)) return *this;

	VLONG	i = 0;

	TYPE	*pbuf = l_ptr;
	TYPE	*tbuf = t.l_ptr;
	for (i = 0; i < count; i++, pbuf++, tbuf++) *pbuf = *tbuf;

	return *this;
}

template<class TYPE> GeDynamicArray<TYPE>& GeDynamicArray<TYPE>::operator =(const TYPE *t)
{
	Free();
	VLONG	cnt = sizeof(*t) / sizeof(TYPE);
	if (!ReScopeNC(cnt)) return *this;

	VLONG	i = 0;

	TYPE	*pbuf = l_ptr;
	const TYPE	*tbuf = t;
	for (i = 0; i < count; i++, pbuf++, tbuf++) *pbuf = *tbuf;

	return *this;
}

template<class TYPE> void GeDynamicArray<TYPE>::SetBlockSize(VLONG size)
{ 
	blocksize = size;
}

//Read array from hyperfile with list of types in order of how they appear in structures
template<class TYPE> Bool GeDynamicArray<TYPE>::Read(HyperFile *hf, GeDynamicArray<LONG> &typelist)
{
	if (!typelist || !hf) return FALSE;

	/*
	//---INTERNAL---

	LONG level = 0;
	if (!hf->ReadLong(&level)) return FALSE;
	if (level == -1) //Level 1 type GeDynamicArray
	{
	LLONG r_count = 0, r_blocksize = 0;
	if (!hf->ReadLLong(&r_count)) return FALSE;
	if (!hf->ReadLLong(&r_blocksize)) return FALSE;

	//Warning!!!
	//Truncates if not on a 64bit system and count is too large!
	count = VLONG(r_count);
	blocksize = VLONG(r_blocksize);
	}
	else
	{
	LONG r_blocksize = 0;
	count = level;

	if (!hf->ReadLong(&r_blocksize)) return FALSE;
	blocksize = r_blocksize;
	}

	//---INTERNAL---
	*/

	LONG r_count = 0, r_blocksize = 0;

	if (!hf->ReadLong(&r_count)) return FALSE;
	if (!hf->ReadLong(&r_blocksize)) return FALSE;

	SetBlockSize(VLONG(r_blocksize));
	if (!ReScopeNC(VLONG(r_count))) return FALSE;
	
	TYPE	*tbuf = l_ptr;
	VLONG	i = 0, o = 0;
	for (i = 0; i < count; i++, tbuf++)
	{
		CHAR	*ttbuf = (CHAR *) tbuf;
		for (o = 0; o < typelist.count; o++)
		{
			switch (typelist[o])
			{
			case DATYPE_CHAR:			if (!hf->ReadChar(ttbuf)) return FALSE; ttbuf += sizeof(CHAR); break;
			case DATYPE_UCHAR:		if (!hf->ReadUChar((UCHAR *) ttbuf)) return FALSE; ttbuf += sizeof(UCHAR); break;
			case DATYPE_WORD:			if (!hf->ReadWord((SWORD *) ttbuf)) return FALSE; ttbuf += sizeof(SWORD); break;
			case DATYPE_UWORD:		if (!hf->ReadUWord((UWORD *) ttbuf)) return FALSE; ttbuf += sizeof(UWORD); break;
			case DATYPE_LONG:			if (!hf->ReadLong((LONG *) ttbuf)) return FALSE; ttbuf += sizeof(LONG); break;
			case DATYPE_ULONG:		if (!hf->ReadULong((ULONG *) ttbuf)) return FALSE; ttbuf += sizeof(ULONG); break;
			case DATYPE_REAL:			if (!hf->ReadReal((Real *) ttbuf)) return FALSE; ttbuf += sizeof(Real); break;
			case DATYPE_LREAL:		if (!hf->ReadLReal((LReal *) ttbuf)) return FALSE; ttbuf += sizeof(LReal); break;
			case DATYPE_BOOL:			if (!hf->ReadBool((Bool *) ttbuf)) return FALSE; ttbuf += sizeof(Bool); break;
			case DATYPE_TIME:			if (!hf->ReadTime((BaseTime *) ttbuf)) return FALSE; ttbuf += sizeof(BaseTime); break;
			case DATYPE_VECTOR:		if (!hf->ReadVector((Vector *) ttbuf)) return FALSE; ttbuf += sizeof(Vector); break;
			case DATYPE_LVECTOR:	if (!hf->ReadLVector((LVector *) ttbuf)) return FALSE; ttbuf += sizeof(LVector); break;
			case DATYPE_MATRIX:		if (!hf->ReadMatrix((Matrix *) ttbuf)) return FALSE; ttbuf += sizeof(Matrix); break;
			case DATYPE_LMATRIX:	if (!hf->ReadLMatrix((LMatrix *) ttbuf)) return FALSE; ttbuf += sizeof(LMatrix); break;
			case DATYPE_STRING:		{ if (!hf->ReadString((String *) ttbuf)) return FALSE; String	 tstring = ((String *) ttbuf)[0]; ttbuf += sizeof(tstring); }break;
			case DATYPE_FILENAME: { if (!hf->ReadFilename((Filename *) ttbuf)) return FALSE; Filename	 tsfn = ((Filename *) ttbuf)[0]; ttbuf += sizeof(tsfn); }break;
			case DATYPE_LLONG:		if (!hf->ReadLLong((LLONG *) ttbuf)) return FALSE; ttbuf += sizeof(LLONG); break;
			default:							ttbuf += typelist[o]; break;
			}
		}
	}
	return TRUE;
}

//Write array to hyperfile
template<class TYPE> Bool	GeDynamicArray<TYPE>::Write(HyperFile *hf, GeDynamicArray<LONG> &typelist)
{
	if (!typelist || !hf) return FALSE;

	//LONG level = -1;
	LONG r_count = LONG(count), r_blocksize = LONG(blocksize);

	//if (!hf->WriteLong(level)) return FALSE;

	if (!hf->WriteLong(r_count)) return FALSE;
	if (!hf->WriteLong(r_blocksize)) return FALSE;
	if (!ptr || count == 0) return TRUE;

	VLONG tsize = sizeof(TYPE);
	if (!tsize) return TRUE;

	VLONG	i = 0, o = 0;
	TYPE	*tbuf = l_ptr;
	for (i = 0; i < count; i++, tbuf++)
	{
		CHAR	*ttbuf = (CHAR *) tbuf;
		for (o = 0; o < typelist.count; o++)
		{
			switch (typelist[o])
			{
			case DATYPE_CHAR:			if (!hf->WriteChar(ttbuf[0])) return FALSE; ttbuf += sizeof(CHAR); break;
			case DATYPE_UCHAR:		if (!hf->WriteUChar(((UCHAR *) ttbuf)[0])) return FALSE; ttbuf += sizeof(UCHAR); break;
			case DATYPE_WORD:			if (!hf->WriteWord(((SWORD *) ttbuf)[0])) return FALSE; ttbuf += sizeof(SWORD); break;
			case DATYPE_UWORD:		if (!hf->WriteUWord(((UWORD *) ttbuf)[0])) return FALSE; ttbuf += sizeof(UWORD); break;
			case DATYPE_LONG:			if (!hf->WriteLong(((LONG *) ttbuf)[0])) return FALSE; ttbuf += sizeof(LONG); break;
			case DATYPE_ULONG:		if (!hf->WriteULong(((ULONG *) ttbuf)[0])) return FALSE; ttbuf += sizeof(ULONG); break;
			case DATYPE_REAL:			if (!hf->WriteReal(((Real *) ttbuf)[0])) return FALSE; ttbuf += sizeof(Real); break;
			case DATYPE_LREAL:		if (!hf->WriteLReal(((LReal *) ttbuf)[0])) return FALSE; ttbuf += sizeof(LReal); break;
			case DATYPE_BOOL:			if (!hf->WriteBool(((Bool *) ttbuf)[0])) return FALSE; ttbuf += sizeof(Bool); break;
			case DATYPE_TIME:			if (!hf->WriteTime(((BaseTime *) ttbuf)[0])) return FALSE; ttbuf += sizeof(BaseTime); break;
			case DATYPE_VECTOR:		if (!hf->WriteVector(((Vector *) ttbuf)[0])) return FALSE; ttbuf += sizeof(Vector); break;
			case DATYPE_LVECTOR:	if (!hf->WriteLVector(((LVector *) ttbuf)[0])) return FALSE; ttbuf += sizeof(LVector); break;
			case DATYPE_MATRIX:		if (!hf->WriteMatrix(((Matrix *) ttbuf)[0])) return FALSE; ttbuf += sizeof(Matrix); break;
			case DATYPE_LMATRIX:	if (!hf->WriteLMatrix(((LMatrix *) ttbuf)[0])) return FALSE; ttbuf += sizeof(LMatrix); break;
			case DATYPE_STRING:		{ if (!hf->WriteString(((String *) ttbuf)[0])) return FALSE; String	 tstring = ((String *) ttbuf)[0]; ttbuf += sizeof(tstring); }break;
			case DATYPE_FILENAME: { if (!hf->WriteFilename(((Filename *) ttbuf)[0])) return FALSE; Filename	 tsfn = ((Filename *) ttbuf)[0]; ttbuf += sizeof(tsfn); }break;
			case DATYPE_LLONG:		if (!hf->WriteLLong(((LLONG *) ttbuf)[0])) return FALSE; ttbuf += sizeof(LLONG); break;
			default:							ttbuf += typelist[o]; break;
			}
		}
	}

	return TRUE;
}

//routine to execute code on all entries in ptr array
template<class TYPE> void GeDynamicArray<TYPE>::DoToAll(GeDynamicArrayCallback *callback, void *data) 
{ 
	if (!ptr) return;

	VLONG	i = 0; 
	TYPE *buf = l_ptr;
	for (i = 0; i < count; i++, buf++)
	{
		callback(i, buf, data); 
	}
}

//routine to execute code on some entries in ptr array
template<class TYPE> void GeDynamicArray<TYPE>::DoToSome(GeDynamicArrayCallback *callback, void *data, VLONG do_start, VLONG do_end)
{
	if (!ptr) return;

	VLONG	dend = do_end;
	VLONG	dsta = do_start;
	if (dend < dsta) dend = do_start, dsta = do_end;
	if (dend > count) dend = count;
	if (dsta < 0) dsta = 0;

	TYPE *buf = &l_ptr[dsta];
	VLONG	i = 0;
	for (i = dsta; i < dend; i++, buf++)
	{
		callback(i, buf, data);
	}
}

//routine to execute code on entries that match TYPE p in ptr array
template<class TYPE> void GeDynamicArray<TYPE>::DoToSearch(GeDynamicArrayCallback *callback, void *data, TYPE p)
{
	if (!ptr) return;

	TYPE *buf = l_ptr;
	VLONG	i = 0;
	for (i = 0; i < count; i++, buf++)
	{
		if (*buf == p) callback(i, buf, data);
	}
}

//Resizes the array without retatining the data and without clearing the data
template<class TYPE> Bool GeDynamicArray<TYPE>::ReScopeNC(VLONG cnt, Bool make_new)
{
	if (cnt <= 0)
	{
		Free();
		return TRUE;
	}

	if (!blocksize)
	{
		GeBoom();
		return FALSE;
	}
	VLONG	nsize = (((cnt + l_cnt - 1) / blocksize) + 1) * blocksize, i = 0;
	TYPE *buf = NULL;

	if (count)
	{
		for (i = 0, buf = l_ptr;i < count;i++, buf++) buf->~TYPE();
	}

	if (nsize != totcnt)
	{
		if (!ptr) ptr = (TYPE*)GeAllocNC(sizeof(TYPE) * nsize);
		else ptr = (TYPE*)GeReallocNC(ptr, sizeof(TYPE) * nsize);

		if (!ptr) return FALSE;

		l_ptr = &ptr[l_cnt];
		totcnt = nsize;
	}

	if (make_new)
	{
		for (i = 0, buf = l_ptr;i < cnt;i++, buf++) new(buf)TYPE;
	}

	count = cnt;
	return TRUE;
}

//Resizes the array without retaining the data
template<class TYPE> Bool	GeDynamicArray<TYPE>::ReScope(VLONG cnt)
{
	if (cnt <= 0)
	{
		Free();
		return TRUE;
	}

	if (!blocksize)
	{
		GeBoom();
		return FALSE;
	}
	VLONG	nsize = (((cnt + l_cnt - 1) / blocksize) + 1) * blocksize, i = 0;
	TYPE *buf = NULL;

	if (count)
	{
		for (i = 0,buf = l_ptr;i < count;i++, buf++) buf->~TYPE();
	}

	if (nsize != totcnt)
	{
		if (!ptr) ptr = (TYPE*)GeAllocNC(sizeof(TYPE) * nsize);
		else ptr = (TYPE*)GeReallocNC(ptr, sizeof(TYPE) * nsize);

		if (!ptr) return FALSE;

		l_ptr = &ptr[l_cnt];

		totcnt = nsize;
	}

	ClearMem(l_ptr, sizeof(TYPE) * cnt);
	for (i = 0, buf = l_ptr;i < cnt;i++, buf++) new(buf)TYPE;

	count = cnt;
	return TRUE;
}

//Resizes the array without clearing the memory used
template<class TYPE> Bool	GeDynamicArray<TYPE>::ReSizeNC(VLONG cnt, Bool make_new)
{
	if (cnt == count) return TRUE;

	if (cnt < count) return Remove(cnt, count - cnt);
	return InsertSpace(count,cnt - count,FALSE, make_new);
}

template<class TYPE> Bool	GeDynamicArray<TYPE>::ReSize(VLONG cnt)
{
	if (cnt == count) return TRUE;

	if (cnt < count) return Remove(cnt, count - cnt);
	return InsertSpace(count,cnt - count,TRUE);
}

//Reize the array down
template<class TYPE> Bool	GeDynamicArray<TYPE>::ReSize(VLONG c_from, VLONG c_to)
{
	VLONG	t_from = VCut(c_from, 0, count), t_to = VCut(c_to + 1, 0, count);
	if (t_from == t_to) return ReSize(0);
	if (t_from > t_to)
	{
		VLONG	temp = t_from;
		t_from = t_to;
		t_to = temp;
	}

	Shift(t_from, -t_from);
	return ReSize(t_to - t_from);
}

//Fill the array with TYPE t between c_from and c_to
template<class TYPE> inline void GeDynamicArray<TYPE>::Fill(VLONG c_from, VLONG c_to, TYPE t)
{
	VLONG	t_from = VCut(c_from, 0, count), t_to = VCut(c_to + 1, 0, count);
	if (t_from == t_to) return;
	if (t_from > t_to)
	{
		VLONG	temp = t_from;
		t_from = t_to;
		t_to = temp;
	}

	TYPE	*buf = NULL;
	VLONG	i = 0;
	for (i = t_from, buf = &l_ptr[t_from]; i < t_to; i++, buf++) *buf = t;
}

//Fill the array with TYPE t
template<class TYPE> inline void GeDynamicArray<TYPE>::Fill(TYPE t)	
{ 
	TYPE	*buf = l_ptr; 
	VLONG i = 0; 
	for (i = 0; i < count; i++, buf++) *buf = t; 
}

//Add value to the array at index
//and shifts array up
template<class TYPE> Bool GeDynamicArray<TYPE>::Insert(TYPE p, VLONG index)
{
	if (index < 0) return FALSE;

	if (!InsertSpace(index,1,TRUE)) return FALSE;
	l_ptr[index] = p;
	return TRUE;
}

//Inserts (copies) an existing array into this one
template<class TYPE> Bool	GeDynamicArray<TYPE>::Insert(const TYPE *p, VLONG index, VLONG cnt)
{
	if (index < 0 || index > count || !p || cnt <= 0) return FALSE;

	if (!InsertSpace(index, cnt, TRUE)) return FALSE;
	VLONG i=0;
	TYPE *buf = &l_ptr[index];
	const TYPE *pbuf = p;
	for (i = 0;i < cnt;i++, pbuf++, buf++) (*buf) = (*pbuf);
	return TRUE;
}

//Merges an existing dynamic array into this one, this array becomes the owner
template<class TYPE> Bool	GeDynamicArray<TYPE>::MergeIn(GeDynamicArray<TYPE>&p,VLONG index)
{
	if (index < 0 || index > count || !p || p.count <= 0) return FALSE;

	if (!InsertSpace(index, p.count, FALSE, FALSE)) return FALSE;
	VLONG cnt = p.count;
	TYPE *data = p.Release();
	if (!data) return FALSE;
	CopyMem(data, &l_ptr[index], sizeof(TYPE) * cnt);
	GeFree(data);
	return TRUE;
}

//Inserts a gap in the array
template<class TYPE> Bool GeDynamicArray<TYPE>::InsertSpace(VLONG index,VLONG amount,Bool clear,Bool makenew)
{
	if (index < 0) return FALSE;

	VLONG	cnt = ((count > index)?count:index) + amount;

	if (!blocksize)
	{
		GeBoom();
		return FALSE;
	}

	if (index < (count >> 1)) //Insert space low
	{
		l_cnt -= amount;
		if (l_cnt < 0) //resize it's larger
		{
			VLONG nadd = (((-l_cnt - 1)/blocksize) + 1) * blocksize;

			totcnt += nadd;
			l_cnt += nadd;

			if (!ptr) ptr = (TYPE*)GeAllocNC(sizeof(TYPE) * totcnt);
			else ptr = (TYPE*)GeReallocNC(ptr, sizeof(TYPE) * totcnt);

			if (!ptr) return FALSE;

			l_ptr = &ptr[l_cnt];

			if (index > 0)
			{
				memmove(&l_ptr[index + amount], &l_ptr[(amount - nadd) + index], sizeof(TYPE)* (count - index));
				memmove(l_ptr, &l_ptr[amount - nadd], sizeof(TYPE) * index);
			}
			else
			{
				memmove(&l_ptr[amount], &l_ptr[amount - nadd], sizeof(TYPE) * count);
			}
		}
		else //just shift space in
		{
			if (index > 0) memmove(&ptr[l_cnt], l_ptr, sizeof(TYPE) * index);
			l_ptr = &ptr[l_cnt];
		}
	}
	else //Insert space high
	{
		VLONG	nsize = (((cnt + l_cnt - 1) / blocksize) + 1) * blocksize;

		if (nsize > totcnt) //Resize it's larger
		{
			if (!ptr) ptr = (TYPE*)GeAllocNC(sizeof(TYPE) * nsize);
			else ptr = (TYPE*)GeReallocNC(ptr, sizeof(TYPE) * nsize);

			if (!ptr) return FALSE;

			l_ptr = &ptr[l_cnt];

			totcnt = nsize;
		} 

		//Shift space in
		if (index < count)
		{
			memmove(&(l_ptr[index + amount]), &(l_ptr[index]),  sizeof(TYPE) * (count - index));
		}
	}

	VLONG i = 0;
	TYPE *buf = NULL;

	if (clear) 
	{
		if (count < index) //Clear space up till index
			ClearMem(&l_ptr[count], sizeof(TYPE) * (index - count), 0);

		//Clear space from index over amount
		ClearMem(&l_ptr[index], sizeof(TYPE) * amount, 0);
	}

	if (makenew)
	{
		if (count < index) //Placement new from end of array to index
			for (i = count, buf = &l_ptr[count]; i<index; i++, buf++) new(buf)TYPE;

		// Placement new over amount
		for (i = 0, buf = &l_ptr[index]; i < amount; i++, buf++) new(buf)TYPE;
	}

	count = cnt;

	return TRUE;
}

//Add value to the end of the array
template<class TYPE> Bool GeDynamicArray<TYPE>::Push(TYPE p)	
{ 
	return Insert(p, count);
}

//Remove value from the array at index
//and shifts array down
template<class TYPE> Bool GeDynamicArray<TYPE>::Remove(VLONG index, VLONG amount) 
{ 
	if (index < 0 || index >= count || !ptr) return FALSE;
	if (!blocksize)
	{
		GeBoom();
		return FALSE;
	}

	if (index <= 0 && amount >= count)
	{
		Free();
		return TRUE;
	}

	VLONG top = VMin(index + amount, count), amt = top - index;
	VLONG i = 0;
	TYPE *buf = NULL;

	for (i = 0, buf = &l_ptr[index];i < amt;i++, buf++) buf->~TYPE();

	if (index < (count >> 1))
	{
		l_cnt += amt;

		if (index > 0) memmove(&l_ptr[amt], l_ptr, sizeof(TYPE) * index); //shift l_ptr up

		l_ptr = &ptr[l_cnt];

		if (l_cnt >= blocksize)
		{
			VLONG nsub = (l_cnt/blocksize) * blocksize;

			totcnt -= nsub;
			l_cnt -= nsub;

			memmove(&l_ptr[-nsub], l_ptr, sizeof(TYPE) * (count - amt)); //Shift down

			ptr = (TYPE*)GeReallocNC(ptr, sizeof(TYPE) * totcnt);
			if (!ptr) return FALSE;

			l_ptr = &ptr[l_cnt];
		}
	}
	else
	{
		VLONG	nsize = ((((count + l_cnt - 1) - amt) / blocksize) + 1) * blocksize;

		if (index < count - 1) memmove(&(l_ptr[index]), &(l_ptr[index + amt]),sizeof(TYPE) * (count - (index + amt) ) );

		if (nsize < totcnt)
		{
			ptr = (TYPE*)GeReallocNC(ptr, sizeof(TYPE) * nsize);
			if (!ptr) return FALSE;

			l_ptr = &ptr[l_cnt];

			totcnt = nsize;
		} 
	}
	count-=amt;
	return TRUE;
}

//Removes all that match p from the array
template<class TYPE> inline void GeDynamicArray<TYPE>::RemoveAll(TYPE p) 
{ 
	VLONG	index = 0; 
	while (TRUE) 
	{ 
		index = Find(p, index); 
		if (index == NOTOK) break; 
		Remove(index); 
	} 
}

//Remove value from the end of the array
template<class TYPE> void GeDynamicArray<TYPE>::Pop(void) 
{ 
	Remove(count - 1);
}

//Find the index of type that matches type p
template<class TYPE> inline VLONG GeDynamicArray<TYPE>::Find(TYPE p)	
{ 
	return Find(p,0); 
}

//Try to find an object from index onwards that matches type p
template<class TYPE> inline VLONG GeDynamicArray<TYPE>::Find(TYPE p, VLONG index)
{
	if (index >= count || !ptr) return NOTOK;

	TYPE	*buf = &l_ptr[index];
	VLONG	i = 0;
	for (i = index; i < count; i++, buf++)
		if (*buf == p) return i;
	return NOTOK;
}

//Shift array down or up from index value
// negative to move down, positive to move up
template<class TYPE> void	GeDynamicArray<TYPE>::Shift(VLONG index, VLONG cnt)
{
	ShiftBlock(index,count,cnt);
}

//Shift a block of the array up or down
// negative to move down, positive to move up
template<class TYPE> void	GeDynamicArray<TYPE>::ShiftBlock(VLONG b_from, VLONG b_to, VLONG amount)
{
	VLONG block_start = VMin(b_from, b_to);
	VLONG block_end = VMax(b_from, b_to);
	if (amount == 0 || block_start == block_end || block_start < 0 || block_end > count) return;

	TYPE	*buf = NULL;
	VLONG	i = 0;
	if (amount > 0)
	{
		VLONG	istart = block_start;
		VLONG	iend = block_end - amount;

		if (iend < istart) iend = istart;

		VLONG m_min = VMin(count, VMax(block_end, istart + amount));
		VLONG m_max = VMin(count, block_end + amount);

		for (i = m_min, buf = &l_ptr[m_min]; i < m_max; i++, buf++) buf->~TYPE();

		m_min = VMin(count, istart + amount);

		if (m_max - m_min > 0)
			memmove(&(l_ptr[m_min]),&(l_ptr[istart]),sizeof(TYPE)*(m_max - m_min));

		iend = VMin(istart + amount, block_end);

		ClearMem(&l_ptr[istart],sizeof(TYPE)*(iend - istart),0);
		for (i = istart, buf = &l_ptr[istart]; i < iend; i++, buf++) new(buf)TYPE;
	}
	else
	{
		VLONG	istart = block_start;
		VLONG	iend = block_end;

		VLONG	m_shift = 0;
		VLONG	m_end = VMax(0, iend + amount), m_max = VMin(m_end, istart);
		VLONG	m_to = istart + amount;
		if ( m_to < 0 )
		{
			m_to = 0;
			m_shift =- (istart + amount);
		}

		for (i = m_to, buf = &l_ptr[m_to]; i < m_max; i++, buf++) buf->~TYPE();

		if (m_end - m_to > 0)
			memmove(&(l_ptr[m_to]),&(l_ptr[istart + m_shift]),sizeof(TYPE)*(m_end - m_to));

		istart = VMax(block_start, m_end);

		ClearMem(&l_ptr[istart],sizeof(TYPE)*(iend - istart),0);
		for (i = istart, buf = &l_ptr[istart]; i < iend; i++, buf++) new(buf)TYPE;
	}
}

//Dynamic Array takes control of a datablock given to it
template<class TYPE> void	GeDynamicArray<TYPE>::Grab(TYPE *p, VLONG amount)
{
	Free();
	ptr = l_ptr = p;
	l_cnt = 0;
	if (ptr)
		totcnt = count = amount;
	else
		totcnt = count = 0;
}

//-----------------------------------------
//Safe Variant, this can be accessed via [] safely and will never get out of bounds
template<class TYPE> class GeSafeDynamicArray : public GeDynamicArray < TYPE >
{
	INSTANCEOF(GeSafeDynamicArray<TYPE>, GeDynamicArray<TYPE>)
public:
	GeSafeDynamicArray(void)	
	{ 
		this->ptr = NULL;
		this->blocksize = 4;
		this->Free();
	}

	GeSafeDynamicArray(VLONG size)
	{ 
		this->ptr = NULL;
		this->blocksize = 4;
		this->Free();
		this->ReScope(size); 
	}

	GeSafeDynamicArray(VLONG size, VLONG block) 
	{
		this->ptr = NULL;
		this->blocksize = block;
		this->Free();
		this->ReScope(size); 
	}

	~GeSafeDynamicArray(void)
	{
		this->Free();
	}

	GeSafeDynamicArray<TYPE> &operator =(const GeSafeDynamicArray<TYPE> &t)
	{
		this->Free();

		this->SetBlockSize(t.GetBlockSize());
		this->ReScopeNC(t.count);

		VLONG	i = 0;
		TYPE *buf = this->l_ptr, *buf2 = t.l_ptr;
		for (i = 0; i < this->count; i++, buf++, buf2++) (*buf) = (*buf2);

		return *this;
	}

	TYPE & operator[](LONG i) const
	{
		if (this->count == 0)
		{
			((GeDynamicArray<TYPE>*)this)->ReScope(1);
			return this->l_ptr[0];
		}
		if (i < 0) return this->l_ptr[0];
		if (i >= this->count) return this->l_ptr[this->count - 1];
		return this->l_ptr[i];
	}
#if defined __LINUX && !defined __C4D_64BIT
	TYPE & operator[](VLONG i) const
	{
		if (this->count == 0)
		{
			((GeDynamicArray<TYPE>*)this)->ReScope(1);
			return this->l_ptr[0];
		}
		if (i < 0) return this->l_ptr[0];
		if (i >= this->count) return this->l_ptr[this->count - 1];
		return this->l_ptr[i];
	}
#endif
	TYPE & operator[](LLONG i) const
	{
		if (this->count == 0)
		{
			((GeDynamicArray<TYPE>*)this)->ReScope(1);
			return this->l_ptr[0];
		}
		if (i < 0) return this->l_ptr[0];
		if (i >= this->count) return this->l_ptr[this->count - 1];
		return this->l_ptr[i];
	}
#if !defined( __LINUX ) && !( defined( __MAC ) && __LP64__ )
	TYPE & operator[] ( int i ) const
	{
		return operator[]((VLONG) i);
	}
#endif
	TYPE & operator[] ( unsigned i ) const
	{
		return operator[]((VLONG) i);
	}
};

//--------------------------------------------------------
//Automatic variant, this will resize based on the index you specify in array[index]

#define DEFAULT_ADR_BLOCK_SIZE 16

template<class TYPE> class GeAutoDynamicArray : public GeDynamicArray < TYPE >
{
	INSTANCEOF(GeAutoDynamicArray<TYPE>, GeDynamicArray<TYPE>)
public:
	GeAutoDynamicArray(void)
	{ 
		this->ptr = NULL;
		this->blocksize = DEFAULT_ADR_BLOCK_SIZE;
		this->Free();
	}

	GeAutoDynamicArray(VLONG size) 
	{ 
		this->ptr = NULL;
		this->blocksize = DEFAULT_ADR_BLOCK_SIZE;
		this->Free();
		this->ReScope(size);
	}

	GeAutoDynamicArray(VLONG size, VLONG block) 
	{ 
		this->ptr = NULL;
		this->blocksize = block;
		this->Free();
		this->ReScope(size);
	}
	~GeAutoDynamicArray(void)
	{
		this->Free();
	}

	GeAutoDynamicArray<TYPE> &operator =(const GeAutoDynamicArray<TYPE> &t)
	{
		this->Free();
		this->SetBlockSize(t.GetBlockSize());
		this->ReScopeNC(t.count);

		VLONG	i = 0;
		TYPE *buf = this->l_ptr, *buf2 = t.l_ptr;
		for (i = 0; i < this->count; i++, buf++, buf2++) (*buf) = (*buf2);

		return *this;
	}

	GeAutoDynamicArray<TYPE> &operator =(TYPE *t) 
	{ 
		this->Free(); 
		this->ptr = t; 
		this->count = sizeof(this->ptr) / sizeof(TYPE); 
		this->totcnt = this->count; 
		this->l_ptr = &(this->ptr[this->l_cnt]);
		return *this; 	
	}

	TYPE & operator[](LONG i) const
	{
		if (this->count == 0)
		{
			((GeDynamicArray<TYPE>*)this)->ReScope(1);
		}
		if (i < 0) return this->l_ptr[0];
		if (i >= this->count)
		{
			((GeDynamicArray<TYPE>*)this)->ReSize(i + 1);
		}
		return this->l_ptr[i];
	}
#if defined __LINUX && !defined __C4D_64BIT
	TYPE & operator[](VLONG i) const
	{
		if (this->count == 0)
		{
			((GeDynamicArray<TYPE>*)this)->ReScope(1);
		}
		if (i < 0) return this->l_ptr[0];
		if (i >= this->count)
		{
			((GeDynamicArray<TYPE>*)this)->ReSize(i + 1);
		}
		return this->l_ptr[i];
	}
#endif
	TYPE & operator[](LLONG i) const
	{
		if (this->count == 0)
		{
			((GeDynamicArray<TYPE>*)this)->ReScope(1);
		}
		if (i < 0) return this->l_ptr[0];
		if (i >= this->count)
		{
			((GeDynamicArray<TYPE>*)this)->ReSize(i + 1);
		}
		return this->l_ptr[i];
	}
#if !defined( __LINUX ) &&  !( defined( __MAC ) && __LP64__)
	TYPE & operator[] ( int i ) const
	{
		return operator[]((VLONG) i);
	}
#endif
	TYPE & operator[] ( unsigned i ) const
	{
		return operator[]((VLONG) i);
	}
};
#endif


