/////////////////////////////////////////////////////////////
// CINEMA 4D SDK                                           //
/////////////////////////////////////////////////////////////
// (c) 1989-2004 MAXON Computer GmbH, all rights reserved  //
/////////////////////////////////////////////////////////////

#ifndef _LIB_MODELLING_H_
#define _LIB_MODELLING_H_

#include "c4d_library.h"
#include "c4d_memory.h"

class iModeling;
class ModelingBase;
class ModelingKernel;
class Ngon;

//////////////////////////////////////////////////////////////////////////

#ifndef NOTINDEX
#define NOTINDEX				(MAXLONGl)
#endif

struct PointMove
{
	PointMove(LONG index) { i=index; }

	LONG	i;
	Vector	p;
};

#define TRANSMAP_FLAG_NEW		(1<<0)
#define TRANSMAP_FLAG_CLONE		(1<<1)
#define TRANSMAP_FLAG_MOVED		(1<<2)
#define TRANSMAP_FLAG_DELETED	(1<<3)
#define TRANSMAP_FLAG_UNION		(1<<4)

#define TRANSMAP_PNTFLAG_INTER	(1<<20)
#define TRANSMAP_PNTFLAG_DELTA	(1<<21)
#define TRANSMAP_PNTFLAG_WELDED	(1<<22)

#define TRANSMAP_PLYFLAG_MELTED	(1<<20)
#define TRANSMAP_PLYFLAG_PGON	(1<<21)

struct TransIndexMapData
{
	TransIndexMapData(LONG oindex) { oIndex=oindex; }
	TransIndexMapData(LONG oindex, LONG nindex) { oIndex=oindex; nIndex=nindex; }

	LONG oIndex;
	LONG nIndex;
};

struct TransMapData : TransIndexMapData
{
	TransMapData(LONG oindex) : TransIndexMapData(oindex) { }
	TransMapData(LONG oindex, LONG nindex) : TransIndexMapData(oindex,nindex)  { }

	LONG lFlags;
	LONG mIndex;
};

struct TransMapNewData
{
	TransMapNewData(LONG nindex) { nIndex=nindex; }

	LONG nIndex;
	LONG mIndex;
};

struct TransPointInterData
{
	LONG p1;
	LONG p2;
	Real t;
};

struct TransPointDeltaData
{
	Vector opnt;
	Vector dlt;
};

class TranslationMaps
{
public:
	TranslationMaps(void)
	{
		ClearMem(this, sizeof(TranslationMaps), 0);
	}

	LONG FindOriginalPoint(LONG index);
	LONG FindOriginalPolygon(LONG index);
	LONG FindOriginalPgon(LONG index);
	LONG FindOriginalSegment(LONG index);

	LONG FindNewPoint(LONG index);
	LONG FindNewPolygon(LONG index);
	LONG FindNewPgon(LONG index);
	LONG FindNewSegment(LONG index);

	LONG FindWeldPoint(LONG index);
	LONG FindMeltPolygon(LONG index);

	//////////////////////////////////////////////////////////////////////////

	LONG m_oPointCount;
	LONG m_oPolygonCount;
	LONG m_oPgonCount;
	LONG m_oSegmentCount;

	LONG m_nPointCount;
	LONG m_nPolygonCount;
	LONG m_nPgonCount;
	LONG m_nSegmentCount;

	LONG m_mPointCount;
	LONG m_mPolygonCount;
	LONG m_mPgonCount;
	LONG m_mSegmentCount;

	LONG m_mWeldCount;
	LONG m_mMeltCount;

	//////////////////////////////////////////////////////////////////////////

	TransMapData		*m_pPointMap;
	TransMapData		*m_pPolygonMap;
	TransMapData		*m_pPgonMap;
	TransMapData		*m_pSegmentMap;

	TransIndexMapData	*m_pWeldMap;
	TransIndexMapData	*m_pMeltMap;

	TransMapNewData		*m_pNewPointMap;
	TransMapNewData		*m_pNewPolygonMap;
	TransMapNewData		*m_pNewPgonMap;
	TransMapNewData		*m_pNewSegmentMap;

	TransPointInterData *m_pInterpolatedPoints;
	TransPointDeltaData	*m_pDeltaPoints;
};

//////////////////////////////////////////////////////////////////////////

#define	MODELING_ERROR_UNKNOWN			(-1)
#define	MODELING_ERROR_NONE				(0)
#define	MODELING_ERROR_FAILED			(1)
#define	MODELING_ERROR_NOMEMORY			(2)
#define	MODELING_ERROR_NOTFOUND			(3)
#define	MODELING_ERROR_INVALIDOP		(4)
#define	MODELING_ERROR_FATAL			(5)
#define	MODELING_ERROR_INVALIDOBJECT	(6)
#define	MODELING_ERROR_ILLEGAL			(7)
#define	MODELING_ERROR_TRIANGULATION	(8)
#define	MODELING_ERROR_INVALIDKERNEL	(9)
#define	MODELING_ERROR_INTERNAL			(10)
#define	MODELING_ERROR_BADARGS			(11)
#define	MODELING_ERROR_CORRUPTOBJECT	(12)
#define	MODELING_ERROR_MSGFAIL			(13)

#define MODELING_COMMIT_NONE		0
#define MODELING_COMMIT_UPDATE		1
#define MODELING_COMMIT_CREATEMAP	2
#define MODELING_COMMIT_REFRESH		4
#define MODELING_COMMIT_TRINGONS	8
#define MODELING_COMMIT_RESTORE		16
#define MODELING_COMMIT_ADDUNDO		32
#define MODELING_COMMIT_QUADS		64
#define MODELING_COMMIT_NOVALIDATION	128
#define MODELING_COMMIT_QUADLIMIT		256
#define MODELING_COMMIT_NO_NGONS  (MODELING_COMMIT_TRINGONS | MODELING_COMMIT_QUADS)

#define MODELING_SETPOINT_FLAG_EMPTY	0
#define MODELING_SETPOINT_FLAG_NODIRTY	1
#define MODELING_SETPOINT_FLAG_UNION	2

#define MODELING_SETNGON_FLAG_EMPTY			0
#define MODELING_SETNGON_FLAG_TRIANGULATE	1
#define MODELING_SETNGON_FLAG_FIXEDQUADS	2
#define MODELING_SETNGON_FLAG_NGONQUADS		4
#define MODELING_SETNGON_FLAG_QUADS			8
#define MODELING_SETNGON_FLAG_UNION			16
#define MODELING_SETNGON_FLAG_NOROTATE		32

#define MODELING_GETNGON_FLAG_EMPTY			0
#define MODELING_GETNGON_FLAG_READONLY		1

//////////////////////////////////////////////////////////////////////////

struct CPolygon;

typedef Bool (*TriangulateHook)(BaseObject *pObj, LONG lNgonID, Ngon *ngon, const Vector *pvPoints, LONG lPointCount, CPolygon *&pPolys, LONG &lPolyCount, Bool &bTriang, void *pData);

//////////////////////////////////////////////////////////////////////////

class Modeling
{
private:
	Modeling();
	~Modeling();

	ModelingBase	*m_pBase;
	ModelingKernel	*m_pKernelCache;
	C4DAtom			*m_pObjectCache;

public:

	static Modeling *Alloc();
	static void Free(Modeling *&p);

	//////////////////////////////////////////////////////////////////////////

	Bool InitArray(AtomArray *objs, LONG flags=0);
	Bool InitObject(C4DAtom *op, LONG flags=0);
	Bool Commit(C4DAtom *op=NULL, LONG flags=0 , BaseObject* cobj = NULL);
	void Release();
	void ReleaseObject(C4DAtom *op);

	//////////////////////////////////////////////////////////////////////////

	// Maps

	Bool GetPointMap(C4DAtom *op, LONG **map, LONG *count);
	Bool GetNgonMap(C4DAtom *op, LONG **map, LONG *count);
	Bool ReleaseMap(C4DAtom *op, LONG *map);
	Bool FreeMap(C4DAtom *op, LONG *map);

	// Points

	LONG AddPoint(C4DAtom *op, const Vector &p);
	LONG ClonePoint(C4DAtom *op, LONG index);
	Bool DeletePoint(C4DAtom *op, LONG index);
	Bool GetPoint(C4DAtom *op, LONG index, Vector *p);
	Bool SetPoint(C4DAtom *op, LONG index, const Vector &p, LONG flags=MODELING_SETPOINT_FLAG_EMPTY);
	Bool SetEdgePoint(C4DAtom *op, LONG index, Real l, LONG flags=MODELING_SETPOINT_FLAG_EMPTY);
	Bool SetPoints(C4DAtom *op, PointMove *pnts, LONG cnt, Bool commit=FALSE, LONG flags=MODELING_SETPOINT_FLAG_EMPTY);
	Bool GetPointInfo(C4DAtom *op, LONG index, LONG &ia, LONG &ib, Real &t);
	Bool GetOriginPoint(C4DAtom *op, LONG index, LONG &ci);
	Bool SetPointInfo(C4DAtom *op, LONG index, LONG ia, LONG ib, Real t);
	Bool SetOriginPoint(C4DAtom *op, LONG index, LONG ci);
	LONG GetPointFlags(C4DAtom *op, LONG index);
	LONG SetPointFlags(C4DAtom *op, LONG index, LONG flags);
	Bool GetPointOrigin(C4DAtom *op, LONG index, Vector &opnt);
	Bool SetPointOrigin(C4DAtom *op, LONG index, Vector opnt, LONG flags);
	Bool SetInterPoint(C4DAtom *op, LONG index, Vector npnt, Real t, LONG flags=MODELING_SETPOINT_FLAG_EMPTY);
	Bool SetInterPoint(C4DAtom *op, LONG index, const Vector &move, const Vector &offset, LONG flags=MODELING_SETPOINT_FLAG_EMPTY);

	// Polygon/Ngon

	LONG NewNgon(C4DAtom *op, LONG flags=MODELING_SETNGON_FLAG_EMPTY);
	LONG AddNgon(C4DAtom *op, const Ngon &ngon, LONG flags=MODELING_SETNGON_FLAG_EMPTY);
	LONG CreateNgon(C4DAtom *op, LONG *padr, LONG cnt, LONG flags=MODELING_SETNGON_FLAG_EMPTY);
	Bool DeleteNgon(C4DAtom *op, LONG index, Bool points);
	LONG CloneNgon(C4DAtom *op, LONG index, LONG flags=MODELING_SETNGON_FLAG_EMPTY);
	Bool GetNgon(C4DAtom *op, LONG index, Ngon *ngon, LONG flags=MODELING_GETNGON_FLAG_EMPTY);
	Bool SetNgon(C4DAtom *op, LONG index, Ngon &ngon, LONG flags=MODELING_SETNGON_FLAG_EMPTY);
	Bool GetNgonNormal(C4DAtom *op, LONG index, Vector *n);
	Bool FlipNgonNormal(C4DAtom *op, LONG index);
	Bool GetOriginNgon(C4DAtom *op, LONG index, LONG &ci);
	Bool SetOriginNgon(C4DAtom *op, LONG index, LONG ci);
	LONG GetNgonFlags(C4DAtom *op, LONG index);
	LONG SetNgonFlags(C4DAtom *op, LONG index, LONG flags);
	Bool ResizeNgon(C4DAtom *op, LONG index, LONG pcnt, LONG scnt);

	// Operations

	Bool WeldPoints(C4DAtom *op, LONG source, LONG dest);
	Bool InsertFacePoint(C4DAtom *op, LONG pa, LONG p1);
	Bool CreateHole(C4DAtom *op, LONG index, const Ngon &ngon);
	Bool CreateHole(C4DAtom *op, LONG index, LONG *pnts, LONG pcnt);

	LONG SplitEdge(C4DAtom *op, LONG p1, LONG p2, Real l);
	Bool DeleteEdge(C4DAtom *op, LONG index, LONG p1, LONG p2);

	Bool MeltEdgeBetween(C4DAtom *op, LONG pa, LONG pb, LONG p1, LONG p2);
	Bool MeltEdge(C4DAtom *op, LONG pa, LONG p1, LONG p2);
	Bool MeltPoint(C4DAtom *op, LONG p);
	Bool MeltNgon(C4DAtom *op, LONG pa);

	LONG SplitPolygon(C4DAtom *op, LONG index, LONG p1, LONG p2);
	Bool MergePolygons(C4DAtom *op, LONG pa, LONG pb);

	Bool IsValidEdge(C4DAtom *op, LONG index, LONG p1, LONG p2);
	Bool IsValidNgon(C4DAtom *op, LONG index);
	Bool IsValidPoint(C4DAtom *op, LONG index);

	LONG *GetEdgeNgons(C4DAtom *op, LONG p1, LONG p2, LONG &pcnt);
	LONG *GetPointNgons(C4DAtom *op, LONG p, LONG &pcnt, Bool clone=TRUE);
	LONG *GetNeighborNgons(C4DAtom *op, LONG index, LONG &pcnt);
	LONG *GetPointEdges(C4DAtom *op, LONG p, LONG &ecnt);

	Bool FindNgon(C4DAtom *op, LONG p1, LONG p2, LONG &index);
	Bool GetEdgeSegment(C4DAtom *op, LONG pa, LONG p1, LONG p2, LONG *seg, LONG *s1, LONG *s2, Bool rel);

	Bool GetOriginalEdgePoints(C4DAtom *op, LONG edge, LONG &p1, LONG &p2);

	Bool IsEdgeDeleted(C4DAtom *op, LONG p1, LONG p2);
	Bool IsPointDeleted(C4DAtom *op, LONG index);
	Bool IsFaceDeleted(C4DAtom *op, LONG index);

	LONG TranslateNgonIndex(C4DAtom *op, LONG index, Bool tovirtual=FALSE);
	LONG TranslatePointIndex(C4DAtom *op, LONG index, Bool tovirtual=FALSE);

	Bool GetFaceSelection(C4DAtom *op, BaseSelect *select, BaseSelect *faceselect, BaseSelect *hidden=NULL);
	Bool GetEdgeSelection(C4DAtom *op, BaseSelect *select, BaseSelect *edgeselect, BaseSelect *hidden=NULL);
	LONG *GetEdgeSelectionArray(C4DAtom *op, BaseSelect *select, LONG &ecnt, BaseSelect *hidden=NULL, Bool tovirtual=TRUE);

	// Information

	LONG GetLastError(C4DAtom *op);
	Bool IsModified(C4DAtom *op);

	// Management

	void FreeTable(C4DAtom *op, void *table);

	// Helper

	Bool PointInFace(C4DAtom *op, LONG index, const Vector &p);
	Bool PointInFace(C4DAtom *op, const Ngon &ngon, const Vector &p);
	Bool LineFaceIntersection(C4DAtom *op, LONG index, const Vector &p1, const Vector &p2);
	Bool LineFaceIntersection(C4DAtom *op, const Ngon &ngon, const Vector &p1, const Vector &p2);

	// Hooks

	void SetTriangulateHook(C4DAtom *op, TriangulateHook pHook, void *pData);
	void GetTriangulateHook(C4DAtom *op, TriangulateHook *ppHook, void **ppData);
};

// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF

//////////////////////////////////////////////////////////////////////////
#define LIBRARY_MODELLING	1015814
//////////////////////////////////////////////////////////////////////////

struct ModelingLib : public C4DLibrary
{
	iModeling *(*Alloc)();
	void (*Free)(iModeling *&p);

	Bool (iModeling::*InitArray)(AtomArray *objs, LONG flags);
	Bool (iModeling::*InitObject)(C4DAtom *op, LONG flags);
	Bool (iModeling::*Commit)(C4DAtom *op, LONG flags, BaseObject* cobj);
	void (iModeling::*Release)();
	void (iModeling::*ReleaseObject)(C4DAtom *op);

	LONG (iModeling::*AddPoint)(C4DAtom *op, const Vector &p);
	LONG (iModeling::*ClonePoint)(C4DAtom *op, LONG index);
	Bool (iModeling::*DeletePoint)(C4DAtom *op, LONG index);
	Bool (iModeling::*GetPoint)(C4DAtom *op, LONG index, Vector *p);
	Bool (iModeling::*SetPoint)(C4DAtom *op, LONG index, const Vector &p, LONG flags);
	Bool (iModeling::*SetEdgePoint)(C4DAtom *op, LONG index, Real l, LONG flags);
	Bool (iModeling::*SetPoints)(C4DAtom *op, PointMove *pnts, LONG cnt, Bool commit, LONG flags);

	LONG (iModeling::*NewNgon)(C4DAtom *op, LONG flags);
	LONG (iModeling::*AddNgon)(C4DAtom *op, const Ngon &ngon, LONG flags);
	LONG (iModeling::*CreateNgon)(C4DAtom *op, LONG *padr, LONG cnt, LONG flags);
	Bool (iModeling::*DeleteNgon)(C4DAtom *op, LONG index, Bool points);
	LONG (iModeling::*CloneNgon)(C4DAtom *op, LONG index, LONG flags);
	Bool (iModeling::*GetNgon)(C4DAtom *op, LONG index, Ngon *ngon, LONG flags);
	Bool (iModeling::*SetNgon)(C4DAtom *op, LONG index, Ngon &ngon, LONG flags);
	Bool (iModeling::*GetNgonNormal)(C4DAtom *op, LONG index, Vector *n);
	Bool (iModeling::*FlipNgonNormal)(C4DAtom *op, LONG index);

	LONG (iModeling::*SplitEdge)(C4DAtom *op, LONG p1, LONG p2, Real l);
	Bool (iModeling::*DeleteEdge)(C4DAtom *op, LONG index, LONG p1, LONG p2);

	Bool (iModeling::*MeltEdgeBetween)(C4DAtom *op, LONG pa, LONG pb, LONG p1, LONG p2);
	Bool (iModeling::*MeltEdge)(C4DAtom *op, LONG pa, LONG p1, LONG p2);
	Bool (iModeling::*MeltPoint)(C4DAtom *op, LONG p);

	LONG (iModeling::*SplitPolygon)(C4DAtom *op, LONG index, LONG p1, LONG p2);
	Bool (iModeling::*MergePolygons)(C4DAtom *op, LONG pa, LONG pb);

	Bool (iModeling::*IsValidEdge)(C4DAtom *op, LONG index, LONG p1, LONG p2);
	Bool (iModeling::*IsValidNgon)(C4DAtom *op, LONG index);
	Bool (iModeling::*IsValidPoint)(C4DAtom *op, LONG index);

	LONG *(iModeling::*GetEdgeNgons)(C4DAtom *op, LONG p1, LONG p2, LONG &pcnt);
	LONG *(iModeling::*GetPointNgons)(C4DAtom *op, LONG p, LONG &pcnt, Bool clone);
	LONG *(iModeling::*GetNeighborNgons)(C4DAtom *op, LONG index, LONG &pcnt);
	LONG *(iModeling::*GetPointEdges)(C4DAtom *op, LONG p, LONG &ecnt);
	Bool (iModeling::*FindNgon)(C4DAtom *op, LONG p1, LONG p2, LONG &index);

	Bool (iModeling::*GetOriginalEdgePoints)(C4DAtom *op, LONG edge, LONG &p1, LONG &p2);
	Bool (iModeling::*IsEdgeDeleted)(C4DAtom *op, LONG p1, LONG p2);

	LONG (iModeling::*TranslateNgonIndex)(C4DAtom *op, LONG index, Bool tovirtual);
	LONG (iModeling::*TranslatePointIndex)(C4DAtom *op, LONG index, Bool tovirtual);

	Bool (iModeling::*GetPointMap)(C4DAtom *op, LONG **map, LONG *count);
	Bool (iModeling::*GetNgonMap)(C4DAtom *op, LONG **map, LONG *count);
	Bool (iModeling::*ReleaseMap)(C4DAtom *op, LONG *map);
	Bool (iModeling::*FreeMap)(C4DAtom *op, LONG *map);

	LONG (iModeling::*GetLastError)(C4DAtom *op);
	Bool (iModeling::*IsModified)(C4DAtom *op);

	Bool (iModeling::*GetPointInfo)(C4DAtom *op, LONG index, LONG &ia, LONG &ib, Real &t);
	Bool (iModeling::*GetOriginPoint)(C4DAtom *op, LONG index, LONG &ci);
	Bool (iModeling::*GetOriginNgon)(C4DAtom *op, LONG index, LONG &ci);

	Bool (iModeling::*SetOriginNgon)(C4DAtom *op, LONG index, LONG ci);
	LONG (iModeling::*GetNgonFlags)(C4DAtom *op, LONG index);
	LONG (iModeling::*SetNgonFlags)(C4DAtom *op, LONG index, LONG flags);

	Bool (iModeling::*SetPointInfo)(C4DAtom *op, LONG index, LONG ia, LONG ib, Real t);
	Bool (iModeling::*SetOriginPoint)(C4DAtom *op, LONG index, LONG ci);
	LONG (iModeling::*GetPointFlags)(C4DAtom *op, LONG index);
	LONG (iModeling::*SetPointFlags)(C4DAtom *op, LONG index, LONG flags);

	Bool (iModeling::*GetPointOrigin)(C4DAtom *op, LONG index, Vector &opnt);
	Bool (iModeling::*SetPointOrigin)(C4DAtom *op, LONG index, Vector opnt, LONG flags);
	Bool (iModeling::*SetInterPointBetween)(C4DAtom *op, LONG index, Vector npnt, Real t, LONG flags);
	Bool (iModeling::*WeldPoints)(C4DAtom *op, LONG source, LONG dest);
	Bool (iModeling::*MeltNgon)(C4DAtom *op, LONG pa);

	Bool (iModeling::*IsPointDeleted)(C4DAtom *op, LONG index);
	Bool (iModeling::*IsFaceDeleted)(C4DAtom *op, LONG index);

	Bool (iModeling::*SetInterPointMove)(C4DAtom *op, LONG index, const Vector &move, const Vector &offset, LONG flags);

	Bool (iModeling::*GetFaceSelection)(C4DAtom *op, BaseSelect *select, BaseSelect *faceselect, BaseSelect *hidden);
	Bool (iModeling::*GetEdgeSelection)(C4DAtom *op, BaseSelect *select, BaseSelect *edgeselect, BaseSelect *hidden);
	LONG *(iModeling::*GetEdgeSelectionArray)(C4DAtom *op, BaseSelect *select, LONG &ecnt, BaseSelect *hidden, Bool tovirtual);
	Bool (iModeling::*InsertFacePoint)(C4DAtom *op, LONG pa, LONG p1);
	Bool (iModeling::*CreateHole)(C4DAtom *op, LONG index, const Ngon &ngon);
	Bool (iModeling::*CreateHoleFromPoints)(C4DAtom *op, LONG index, LONG *pnts, LONG pcnt);
	Bool (iModeling::*ResizeNgon)(C4DAtom *op, LONG index, LONG pcnt, LONG scnt);
	Bool (iModeling::*GetEdgeSegment)(C4DAtom *op, LONG pa, LONG p1, LONG p2, LONG *seg, LONG *s1, LONG *s2, Bool rel);

	void (iModeling::*FreeTable)(void *table);

	Bool (iModeling::*PointInFace)(C4DAtom *op, LONG index, const Vector &p);
	Bool (iModeling::*PointInFaceNgon)(C4DAtom *op, const Ngon &ngon, const Vector &p);
	Bool (iModeling::*LineFaceIntersection)(C4DAtom *op, LONG index, const Vector &p1, const Vector &p2);
	Bool (iModeling::*LineFaceIntersectionNgon)(C4DAtom *op, const Ngon &ngon, const Vector &p1, const Vector &p2);

	void (iModeling::*SetTriangulateHook)(C4DAtom *op, TriangulateHook pHook, void *pData);
	void (iModeling::*GetTriangulateHook)(C4DAtom *op, TriangulateHook *ppHook, void **ppData);
};

// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF

#endif
